/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/*-------------------------------------------------------------------------
 * FILE: ladata.c
 * Author: Tilmann Wendel, 05.06.96 BID R&D
 * This file contains the functions for the tracememory and observer
 * functions.
 * Functions to control and set the several trigger are also provided here.
 *-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------
 * Performance Board functions added:
 * Robert Siegmund, 10.03.97 BID R&D
 *-------------------------------------------------------------------------*/

#if defined(_MSC_VER)
# pragma data_seg("ladata_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
#endif /* _MSC_VER etc. */

#define UPLOAD_RETRY_HACK

#ifdef BEST_FIRMWARE

#include <capitype.h>
#include <bios_malloc.h>

static void *malloc(b_int32 n)
{
  void *ptr;
  HWMalloc(&ptr, n);
  return ptr;
}

static void free(void *ptr)
{
  HWFree(ptr);
}


#else /* BEST_FIRMWARE */

#include "lasterr.h"
#include <typedefs.h>
#include <mini_api.h>
#include "b_cmd.h"

#endif /* BEST_FIRMWARE */

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <ladata.h>
#include <regconst.h>
#include <regx30.h>
#include <regx31.h>
#include <regx32.h>
#include <regx33.h>
#include <session.h>
#include <version.h>

#define FAST_HOST_THRESHOLD 10

static char tmp_ladata_buf[1024];


/* ------------------------------------------------------------------------
 * Textual explanation to given error number
 * ------------------------------------------------------------------------ */

/* workaround for missing onboard error reporting capabilites */

#ifndef E2921A_MIN
static b_errtype err;
extern char perfboard_err[127];
#endif


#define STOP_MODE  0
#define RUN_MODE   1
#define READ_MODE  2
#define HEART_BEAT_MODE       3
#define NORMAL_TRIGGER_MODE   4



/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------
 * PCI Trace & External Trigger Functions
 * ------------------------------------------------------------------------
 * ------------------------------------------------------------------------ */

#ifndef E2921A_MIN

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestTracePropDefaultSet(
    b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i;
  b_param_infotype *ParamInfo;
  b_generic_infotype *GenInfo;
  /* license checking */
  B_TRY_BEGIN
  {
    B_TRY_LICENSE(B_CAPABILITY_ANALYZER);

    B_TRY(BestGenInfoGet(handle, B_PARAM_TRACE_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_TRACE_GEN, i, &ParamInfo,
          (b_int32) B_INDEX_SEARCH));

      /* Set it to default */
      B_TRY(BestTracePropSet(handle, ParamInfo->proptyp.traceprop,
          ParamInfo->defaultval));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* ------------------------------------------------------------------------
 * This function sets the properties for the tracememory counter.
 * ------------------------------------------------------------------------ */

b_errtype EXPORT BestTracePropSet(
    b_handletype handle,
    b_traceproptype traceprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestTracePropSet [trcprpset]");

  b_int8 zw;
  b_int32 cmdbuf[3];
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* dynamic CAPABILITY_ checking concerning values of parameters */
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_TRACE_GEN, (b_int32) traceprop, value));

  if (Best16BitRegisterFile(handle))
  {
    B_ERRETURN(BestAbstractPropLongSet(handle, CMD_TRC_PROP_SET, (b_int8) traceprop, value));
  }
  else
  {
    switch (traceprop)
    {
    case B_TRC_HEARTBEATMODE:
      if (value == B_HBMODE_ON)
        zw = HEART_BEAT_MODE;
      else
        zw = NORMAL_TRIGGER_MODE;
      B_ERRETURN(BestBasicBlockWrite(handle, TRC_CMD, (b_int8ptr) & zw, 1UL));

    case B_TRC_HEARTBEATVALUE:
      B_FCT_PARAM_CHK(3, value > 0xffff);
      B_ERRETURN(BestBasicBlockWrite(handle, HEARTBEAT_VALUE, (b_int8ptr) & value, 4UL));

    case B_TRC_TRIG_HISTORY:
      /* special hack to fix a E2925A option #100 shortcoming */
      if (value > 0xffff0UL)
        value = 0xffff0UL;
      B_ERRETURN(BestBasicBlockWrite(handle, ANA_TRIGGER_POINT, (b_int8ptr) & value, 4UL));

    case B_TRC_PERFANALYZER_MODE:
    case B_TRC_PERFPATT0_MODE:
    case B_TRC_TRIG15IO_MODE:
      {
        cmdbuf[0] = 24UL;       /* command = SET_EXTENDED_TRACEPROP */
        cmdbuf[1] = traceprop;
        cmdbuf[2] = value;
        err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmdbuf, 12UL);

        /* if onboard error has been detected, receive error message string */

        if (err != B_E_OK)
        {
          err = BestBasicBlockRead(handle, PERF_DATA, (b_int8ptr) perfboard_err, 127UL);
          if (err != B_E_OK)
          {
            B_ERRETURN(err);
          }

          else
          {
            B_ERRETURN(B_E_PERFBOARD_ERROR);
          }
        }
        else
        {
          B_ERRETURN(B_E_OK);
        }
      }

    default:
      B_FCT_PARAM_ERROR(2, "Invalid property");
    }
  }
}


/*****************************************************************************/
/* This call gets a generic trigger properties back (integer based)          */
/*                                                                           */
/*****************************************************************************/

b_errtype EXPORT BestTracePropGet(b_handletype handle,
    b_traceproptype traceprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME("BestTracePropGet [trcprpget]");

  /* TODO: check parameter */
  /* property value                       */
  b_int8 zw[IN_TRC_PROPGET];
  b_int32 zwl;
  b_int16 outsize;
  b_int8 outbuf[OUT_TRC_PROPGET];
  /* first check if handle is good and check license                    */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    /* New 2x protocol */
    zwl = (b_int32) traceprop;
    (void) BestLong2Stream(zw, &zwl, 1UL);  /* copy the property in the
                                             * stream */
    outsize = OUT_TRC_PROPGET;
    err = BestBasicCommand(handle, CMD_TRC_PROPGET,
      zw, IN_TRC_PROPGET, outbuf, &outsize);
    (void) BestStream2Long(value, outbuf, 1UL);
  }

  B_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function set the equation the trigger/int trigger/cond start look for
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTracePattPropSet(
    b_handletype handle,
    b_tracepattproptype traceprop,
    b_charptrtype pattern
)
{
  B_DECLARE_FUNCNAME("BestTracePattPropSet [trcpprpset]");

  char *ch = "";
  char sync = '\1';
  b_int8ptr zw;                 /* the buffer for the string */
  b_int8ptr bp;
  b_int32 zwl;
  b_int32 length;


  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(pattern);

  if (traceprop != B_PT_TRIGGER &&
    traceprop != B_PT_SQ)
  {
    /* This check is currently not in dynamic capabilities */
    B_FCT_PARAM_ERROR(2, "Invalid property");
  }

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */

    /* save string in case of a syntax error */
    BestLastErrorParamStringSet(handle, pattern);

    /* compute new length of the data stream */
    /* size of traceprop (sizeof(b_int32) = 4) + strlen (input string) + 1
     * for terminating '/0' */
    length = 5UL + (b_int32) strlen(pattern);
    zw = (b_int8ptr) malloc((size_t) length);
    zwl = (b_int32) traceprop;
    bp = BestLong2Stream(zw, &zwl, 1UL);  /* copy the property */
    (void) BestByteCopy(bp, (b_int8ptr) pattern,
      (b_int32) strlen(pattern) + 1UL); /* and pattern */

    /* we assume that the string length conversion to 16 bit is OK */
    err = BestBasicCommandVariable(handle, CMD_TRC_PATTPROPSET,
      zw, (b_int16) length, NULL, NULL);

    free(zw);                   /* free the allocated memory */
    B_ERRETURN(err);
  }
  else
  {
    /* first thing we need to do is to transfer the pattern string to the
     * onboard buffer (since we dont know yet if performance hardware is
     * present and which hardware we are going to use) */

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) & sync, 1UL));
    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) pattern,
        (b_int32) strlen(pattern) + 1UL));

    /* OK, we did all we needed for performance option support so far */

    switch (traceprop)
    {
    case B_PT_TRIGGER:          /* the Trigger is meant */
      B_ERRCHECK(BestBasicBlockWrite(handle, ANA_PATTERN, (b_int8ptr) ch, 1UL));
      B_ERRETURN(BestBasicBlockWrite(handle, ANA_PATTERN, (b_int8ptr) pattern,
          (b_int32) strlen(pattern) + 1UL));

    case B_PT_SQ:               /* the Sample Qualifier is meant */
      B_ERRCHECK(BestBasicBlockWrite(handle, SQ_PATTERN, (b_int8ptr) ch, 1UL));
      B_ERRETURN(BestBasicBlockWrite(handle, SQ_PATTERN, (b_int8ptr) pattern,
          (b_int32) strlen(pattern) + 1UL));

    default:
      B_FCT_PARAM_ERROR(2, "Invalid property");
    }                           /* switch traceprop */
  }
}


/* ------------------------------------------------------------------------
 * This function makes the analyzer run.
 *
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTraceRun(
    b_handletype handle
)
{
  B_DECLARE_FUNCNAME("BestTraceRun [trcrun]");

  b_int8 zw = 1;
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    err = BestBasicCommand(handle, CMD_TRC_RUN, NULL, 0, NULL, NULL);
  }
  else
  {
    err = BestBasicBlockWrite(handle, TRC_CMD, &zw, 1UL);
  }
  B_ERRETURN(err);
}



/* ------------------------------------------------------------------------
 * This function stops the analyzer and switches to readmode.
 *
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTraceStop(
    b_handletype handle
)
{
  B_DECLARE_FUNCNAME("BestTraceStop [trcstop]");

  b_int8 zw = STOP_MODE;
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    err = BestBasicCommand(handle, CMD_TRC_STOP, NULL, 0, NULL, NULL);
  }
  else
  {
    B_ERRCHECK(BestBasicBlockWrite(handle, TRC_CMD, &zw, 1UL));
    zw = READ_MODE;
    err = BestBasicBlockWrite(handle, TRC_CMD, &zw, 1UL);
  }
  B_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function obtains the status from the board. Status means either
 * the pure status info, the trigger point or the number of lines captured.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTraceStatusGet(
    b_handletype handle,
    b_tracestatustype tracestatus,
    b_int32 * status
)
{
  B_DECLARE_FUNCNAME("BestTraceStatusGet [tsget]");

  b_errtype errt;
  b_int8 zw8;
  b_int8 in_zw[IN_TRC_STATUS];
  b_int8 out_zw[OUT_TRC_STATUS];
  b_int32 zw;
  b_int16 addr;

  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(status);

  if (tracestatus != B_TRC_STAT &&
    tracestatus != B_TRC_TRIGPOINT &&
    tracestatus != B_TRC_LINESCAPT)
  {
    /* This check is currently not in the dynamic capabilities */
    B_FCT_PARAM_ERROR(2, "Invalid property");
  }

  if (Best16BitRegisterFile(handle))  /* distinguish old and new protocol */
  {
    /* New 2x protocol */
    zw = tracestatus;
    addr = OUT_TRC_STATUS;
    (void) BestLong2Stream(in_zw, &zw, 1UL);  /* into the bytestream */
    errt = BestBasicCommand(handle, CMD_TRC_STAT_READ,
      in_zw, IN_TRC_STATUS, out_zw, &addr);
    (void) BestStream2Long(status, out_zw, 1UL);
  }
  else
  {

    switch (tracestatus)
    {
    case B_TRC_STAT:
      errt = BestBasicBlockRead(handle, ANA_STATUS, &zw8, 1UL);
      *status = (b_int32) zw8;
      break;

    case B_TRC_TRIGPOINT:
      errt = BestBasicBlockRead(handle, ANA_TRIGGER_POINT,
        (b_int8ptr) status, 4UL);
      break;

    case B_TRC_LINESCAPT:
      errt = BestBasicBlockRead(handle, ANA_LINES_CAPT, (b_int8ptr) status, 4UL);
      break;

    default:
      B_FCT_PARAM_ERROR(2, "Invalid Trace Status");
    }
  }

  B_ERRETURN(errt);
}



/* ------------------------------------------------------------------------
 * This function gets the data and fills them into the array of longs
 * passed down. the user is responsible for allocating enough memory
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestTraceDataGet(
    b_handletype handle,
    b_int32 startline,
    b_int32 n_of_lines,
    b_int32 HUGE * data
)
{
  B_DECLARE_FUNCNAME("BestTraceDataGet [trcdget]");

  b_int8ptr bp;
  b_int8 zw[IN_TRC_DATA];
  b_int16 outsize;
  b_int32 bytesread;
  b_int32 bytes_per_line, lines_per_block;
#ifdef UPLOAD_RETRY_HACK
  int retrycounter = 0;	/* make sure we don't do this too often */
#endif  /* UPLOAD_RETRY_HACK */

  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (n_of_lines)
  {
    /* We allow data==NULL if n_of_lines==0 */
    B_FCT_PARAM_NULL_POINTER_CHK(data);
  }
  else
  {
    B_ERRETURN(B_E_OK);
  }

  if (Best16BitRegisterFile(handle))
  {
    
    /* TODO: CAPI property ? */
    b_porttype port;
    B_ERRCHECK(BestGetPortFromHandle(handle, &port));

    B_ERRCHECK(BestTraceBytePerLineGet(handle, &bytes_per_line));

    switch (port)
    {
      /* try this now case B_PORT_RS232: lines_per_block = 100; break; */

#if 0
     case B_PORT_PCI_IO:
     case B_PORT_PCI_CONF:
      lines_per_block = 0x8000;
      break;

#endif

#ifdef WIN95
     case B_PORT_RS232:
      lines_per_block = FAST_HOST_THRESHOLD;
      break;
#endif

     default:
      lines_per_block = n_of_lines;
      break;
    } /*lint !e788 not all ports used */

    while (n_of_lines)
    {
      if (lines_per_block > n_of_lines)
        lines_per_block = n_of_lines;

      if (lines_per_block > FAST_HOST_THRESHOLD)
      {

#ifdef UPLOAD_RETRY_HACK
/* this special hack is for the RS232 port which looses data with
   some video cards (!! no joke !!)
   Solution: Just reload data that has been lost! (adapt bytesread variable)
*/
	b_int32 startaddr = startline * bytes_per_line;
	
        bytesread = lines_per_block * bytes_per_line;

	err = (BestBasicBlockCommand(handle, B_BIO_TRACEMEM,
				     startaddr,
				     bytesread,
				     bytes_per_line,
				     CMD_READ, (b_int8ptr) data));

	/* error code is B_E_ERROR in release build, only when first
	   error parameter set to B_EERR_PARTIAL_RCVE.

	   in debug case, error code is B_E_PARTIAL_RCVE */
	
	if ( (err == B_E_ERROR || err == B_E_PARTIAL_RCVE) &&
	     retrycounter++ < 20 )
	{
	  b_int64 actErrorCode, numBytes, errpar3, errpar4, errpar5;
	  BestLastErrorAllParamsGet (handle,
				     &actErrorCode, &numBytes,
				     &errpar3, &errpar4, &errpar5);
	  /* adjust so it will fit on a 20byte boundary */

	  if (err == B_E_ERROR && actErrorCode != B_EERR_PARTIAL_RCVE)
	  {
	    B_ERRETURN (err);	/* not our retry-case! */
	  }

#  ifdef BEST_DEBUG
	  /* make block for dbg-out */
	  {
	    char buf[50];
	    sprintf(buf, "\nTrace Data Retry Countdown %d\n"
		    "addr: %ld, bytes read: %ld", retrycounter, startaddr,
		    actErrorCode);
	    DBG_PRINT (buf);
	  }
#  endif
	  /* set read bytes to zero; retry.
	     Every 3rd time decrease buffer size by two */
	  bytesread = 0;
	  if (retrycounter % 3 == 0)
	    lines_per_block /= 2;
	  err = B_E_OK;

	  continue;		/* do it again */
	}
	else if (err == B_E_OK)
	{
	  retrycounter = 0;	/* we made it */
	}
	else
	{
	  B_ERRCHECK(err);
	}


#else  /* not upload_retry_hack */
	bytesread = lines_per_block * bytes_per_line;
        B_ERRCHECK(BestBasicBlockCommand(handle, B_BIO_TRACEMEM,
					 startline * bytes_per_line,
					 bytesread,
					 bytes_per_line,
					 CMD_READ, (b_int8ptr) data));
#endif
	switch (port)
        {
	 case B_PORT_PCI_IO:
	 case B_PORT_PCI_CONF:
          /* PCI driver already swaps data */
          break;

	 default:
          (void) BestStream2Long(data,
				 (b_int8ptr) data,
				 (bytesread) / 4);
          break;
        } /*lint !e788 not all ports used */
      }

      else
      {                         /* the standard mode (the so called "slow"
                                 * mode without fast host */
        bp = BestLong2Stream(zw, &startline, 1);
        (void) BestLong2Stream(bp, &lines_per_block, 1);
        outsize = OUT_TRC_DATA;
        err = BestBasicCommand(handle, CMD_TRC_DATA_READ, zw, IN_TRC_DATA,
			       (b_int8ptr) data, &outsize);

        bytesread = outsize;

        if (err)                /* do it only if no error occured */
          break;

        (void) BestStream2Long(data, (b_int8ptr) data, outsize / 4);
      }

      data += (bytesread / 4);
      n_of_lines -= (bytesread / bytes_per_line);
      startline += (bytesread / bytes_per_line);
    }
  }

  else	/* we are a 25a card! */
  {
    b_int32 startaddr = startline;

    B_ERRCHECK(BestTraceBytePerLineGet(handle, &bytes_per_line));

    while (n_of_lines)
    {
      B_ERRCHECK(BestBasicBlockWrite(handle, ANA_ADDR_PTR,
				     (b_int8ptr) & startaddr, 4UL));

      bytesread = n_of_lines * bytes_per_line;
      
      err = BestBasicBlockRead(handle, ANA_DATA, (b_int8ptr) data, bytesread);
      
#ifdef UPLOAD_RETRY_HACK
      /* error code is B_E_ERROR in release build, only when first
	 error parameter set to B_EERR_PARTIAL_RCVE.
	 in debug case, error code is B_E_PARTIAL_RCVE */
    
      if ( (err == B_E_ERROR || err == B_E_PARTIAL_RCVE) &&
	   retrycounter++ < 20 )
      {
#if 0
	b_int32 actErrorCode, numBytes, errpar3, errpar4, errpar5;
	BestLastErrorAllParamsGet (handle,
				   &actErrorCode, &numBytes,
				   &errpar3, &errpar4, &errpar5);

	if (err == B_E_ERROR && actErrorCode != B_EERR_PARTIAL_RCVE)
	{
	  B_ERRETURN (err);	/* not our retry-case! */
	}

#  ifdef BEST_DEBUG
	/* make block for dbg-out */
	{
	  char buf[50];
	  sprintf(buf, "\nTrace Data Retry Countdown %d\n"
		  "addr: %ld, bytes read: %ld", retrycounter, startaddr,
		  actErrorCode);
	  DBG_PRINT (buf);
	}
#  endif
#endif
	/* set read bytes to zero; retry. */
	bytesread = 0;

	err = B_E_OK;		/* pretend that everything's fine... */
	B_ERRCHECK(BestConnect(handle)); /* basiccommand disconnects on error */

	continue;		/* do it again */
      }
      else if (err == B_E_OK)
      {
	retrycounter = 0;	/* we made it! */
      }
      else
      {
	B_ERRCHECK(err);
      }

      data += (bytesread / 4);
      n_of_lines -= (bytesread / bytes_per_line);
      startline += (bytesread / bytes_per_line);
#endif
      
    }
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This function returns the number of bytes per tracememory line            */
/* The information is taken out of the dynamic capabilities                  */
/*****************************************************************************/
b_errtype EXPORT BestTraceBytePerLineGet(
    b_handletype handle,
    b_int32 * bytes_per_line
)
{
  b_generic_infotype *geninfo;  /* holds the generic infotype for tracememory */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* get the generic information about observer rules */
  B_ERRCHECK(BestGenInfoGet(handle, B_PARAM_TRACE_MEM, &geninfo));

  *bytes_per_line = geninfo->num_pages;

  B_ERRETURN(B_E_OK);
}


/*****************************************************************************/
/* This function returns the bitposition fo a signal and the width of the    */
/* signal in bits.                                                           */
/* The information is taken out of the dynamic capabilities                  */
/*****************************************************************************/
b_errtype EXPORT BestTraceBitPosGet(
    b_handletype handle,
    b_signaltype signal,
    b_int32 * position,
    b_int32 * length
)
{
  b_param_infotype *parinfo;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  *position = 0;
  *length = 0;

  err = BestParamInfoGet(handle, B_PARAM_TRACE_MEM, (b_int32) signal, &parinfo,
    (b_int32) B_ENUM_SEARCH);

  if (err != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32) B_PARAM_TRACE_MEM,
      (b_int32) signal, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  *position = parinfo->min_val; /* we found the signal and here is pos */
  *length = parinfo->max_val;   /* and here the width */

  B_ERRETURN(B_E_OK);
}


/* ------------------------------------------------------------------------
 * This function returns the size of the Trace Memory from the board
 * E2926A ONLY!!! The e2925a does not have flexible trace memory sizes.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestTraceMemorySizeGet(
    b_handletype handle,
    b_int32 * size
)
{
  b_int16 outsize;
  b_int8 outbuf[OUT_TRC_MEMSIZEGET];
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (BestIs2925(handle))
  {
    err = B_E_NOT_E2925A;
  }
  else
  {
    outsize = OUT_TRC_MEMSIZEGET;
    err = BestBasicCommand(handle, CMD_TRC_MEMSIZEGET,
      NULL, IN_TRC_MEMSIZEGET, outbuf, &outsize);
    (void) BestStream2Long(size, outbuf, 1UL);
  }

  B_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function calculates the gap information and fills the array given
 * from the calling function.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestGapInfo(
    b_handletype handle,
    b_int32 num_samples,
    b_int32 HUGE * data
)
{
  b_int32 help1;
  b_int16 gapinfo[100];         /* we make this register a word register ! */
  b_int32 i, index, dataindex;
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  help1 = 0L;

  /* set the address pointer to 0. we start counting at the beginning */
  B_ERRCHECK(BestBasicBlockWrite(handle, ANA_ADDR_PTR, (b_int8ptr) & help1, 4UL));

  /* first zero the complete array */
  for (i = 0L; i < num_samples; i++)
    data[(size_t) i] = 0L;

  dataindex = 0L;
  for (;;)
  {
    index = 0;
    B_ERRCHECK(BestBasicBlockRead(handle, ANA_GAP, (b_int8ptr) gapinfo, 200UL));

    while ((index < 100) && gapinfo[(size_t) index])
    {
      if (dataindex + gapinfo[(size_t) index] + 1L < num_samples)
      {
        dataindex += gapinfo[(size_t) index];
        data[(size_t) (dataindex - 1)] = 1; /* fix the off by one error */
        dataindex++;
      }
      else
        break;

      index++;
    }
    if (index < 100)
      break;
  }

  B_ERRETURN(B_E_OK);
}


b_errtype EXPORT BestGapModeGet(b_handletype handle, b_int32 * mode)
{
  b_int32 analyzer_stat = 0;
  err = BestTraceStatusGet(handle, B_TRC_STAT, &analyzer_stat);

  if (analyzer_stat & 0x4)
  {
    *mode = 1;
  }
  else
  {
    *mode = 0;
  }

  B_ERRETURN(err);
}
#endif
#endif  /* E2921A_MIN */



/* -----------------------------------------------------------------------
 * -----------------------------------------------------------------------
 * PCI Analyzer Run Functions
 * -----------------------------------------------------------------------
 * ----------------------------------------------------------------------- */
#ifndef E2921A_MIN

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestAnalyzerRun(
    b_handletype handle
)
{
  b_int8 zw = 1;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    B_ERRCHECK(BestObsRun(handle));
    B_ERRETURN(BestTraceRun(handle));
  }
  else
  {
    B_ERRETURN(BestBasicBlockWrite(handle, ANA_CMD, &zw, 1UL));
  }


/*   if ((err=BestObsRun(handle))!=B_E_OK) return err; */
/*   return BestTraceRun(handle); */
}


b_errtype EXPORT BestAnalyzerStop(
    b_handletype handle
)
{
  b_int8 zw = STOP_MODE;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    B_ERRCHECK(BestObsStop(handle));
    B_ERRETURN(BestTraceStop(handle));
  }
  else
  {
    B_ERRCHECK(BestBasicBlockWrite(handle, ANA_CMD, &zw, 1UL));
    zw = READ_MODE;
    B_ERRETURN(BestBasicBlockWrite(handle, ANA_CMD, &zw, 1UL));
  }
}

#endif  /* BEST_FIRMWARE */

/*****************************************************************************/
/* This call programs the selected pattern term.                             */
/*****************************************************************************/
b_errtype EXPORT BestPattSet(b_handletype handle,
    b_int32 pattern_ident,
    b_charptrtype pattern)
{
  B_DECLARE_FUNCNAME("BestPattSet [pattset]");

  b_int32 length;
  char sync = '\1';             /* start of string marker                    */
  b_int32 cmd_buf[2];           /* command buffer                            */
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(pattern);
  B_FCT_PARAM_CHK(pattern_ident, pattern_ident > B_PATTERN_SIZE);


  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */

    length = (b_int32) strlen(pattern) + 1UL;

    err = BestBasicCommandVariable(handle,
      CMD_MERGE(CMDG_PATT_SET + pattern_ident, CMDM_PATT_SET),
      (b_int8ptr) pattern, (b_int16) length,
      NULL, NULL);

  }
  else
  {
    cmd_buf[0] = 6UL;           /* this is defined as PROG_PATTERNTERM     */
    cmd_buf[1] = pattern_ident; /* pattern term identifier                 */

    /* first transfer pattern string via the performance data port           */

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) & sync, 1UL));
    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) pattern,
        (b_int32) strlen(pattern) + 1UL));

    /* then send command and property parameter to performance hardware     */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL);

    /* if onboard error has been detected, receive error message string      */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err == B_E_OK)
        err = B_E_PERFBOARD_ERROR;
    }
  }

  B_ERRETURN(err);
}

#endif  /* E2921A_MIN */


b_errtype BestPattGetEdit(b_handletype handle, b_int64 patt_id, b_ccharptrtype * pattern)
{
  B_DECLARE_FUNCNAME("BestPattGetEdit");

  b_int8 pid = (b_int8) patt_id;
  b_int16 out_size = OUT_PATT_GET;
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    B_TRY_LICENSE(B_CAPABILITY_ANALYZER);
    B_TRY_FCT_PARAM_NULL_POINTER(pattern);
    B_TRY_FCT_PARAM(patt_id, patt_id > B_PATTERN_SIZE);
    B_TRY(BestBasicCommand(handle, CMD_PATT_EDIT_GET,
        &pid, IN_PATT_GET,
        (b_int8ptr) tmp_ladata_buf, &out_size));
    *pattern = tmp_ladata_buf;
  }

  B_ERRETURN(B_TRY_RET);
}


/* ------------------------------------------------------------------------
 * This function calculates the gap information and fills the array given
 * from the calling function.
 * ------------------------------------------------------------------------ */

#ifndef E2921A_MIN

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestTraceMemoryFileDump(
    b_handletype handle,
    b_charptrtype filename
)
{
  B_DECLARE_FUNCNAME("BestTraceMemoryFileDump [trcdump]");

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(filename);
  B_ERRETURN(BestTraceMemoryFileDumpAdv(handle,
					filename, 
					~0UL, 
					~0UL, 
					NULL, 
					0,
					B_WFM_STANDARD)); /* not compressed */
}


/* this is the enhanced version of the trace dump. You can specify the number
   of lines before and after the trigger point. The cb will be called
   periodically with an value between 1 and 100 to indicate its progress.
   If it returns with a value of zero the dumping will continue. In any other
   case the filedump will be cancelled. The file keeps existing. */
/* the modebit reflects if the file is to be saved with performance mode or
   not - in the NO case, do not do performance uploads */
/* it is now also possible to specify the compression method (saving disks) */

b_errtype EXPORT BestTraceMemoryFileDumpAdv(
    b_handletype handle,
    b_charptrtype filename,
    b_int32 pretrig,
    b_int32 posttrig,
    int (*cb) (int),
    int modebit,int storemode)
{
  B_DECLARE_FUNCNAME("BestTraceMemoryFileDumpAdv");

  FILE *fp;
  b_int32 lcnt, length, dstart;
  b_int32ptr data;
  b_int32 i, j;
  b_int32 longs_per_line;
  b_int32 bytes_per_line;
  b_int32 status32, triggerpoint;
  b_int32 bus_width;
  int bound;
  /* b_hwtype hwtype; */              /* holds the hardware code */
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(filename);

  /* read the number of bytes per line  */
  B_ERRCHECK(BestTraceBytePerLineGet(handle, &bytes_per_line));

  /* SCR;lint caught this */
  B_FCT_PARAM_CHK(2, bytes_per_line == 0);

  /* this is a shortcut and needs to get fixed soon!!! */
  longs_per_line = bytes_per_line / sizeof(unsigned long);

  B_ERRCHECK(BestTraceStatusGet(handle, B_TRC_STAT, &status32));
  B_ERRCHECK(BestTraceStatusGet(handle, B_TRC_TRIGPOINT, &triggerpoint));
  B_ERRCHECK(BestTraceStatusGet(handle, B_TRC_LINESCAPT, &length));

  /* detect the used hardware type */
  /* B_ERRCHECK(BestGetHWFromHandle(handle, &hwtype)); */

  /* detect the buswidth */
  if (BestHasIchiban(handle))
  {
    B_ERRCHECK(BestSystemInfoGet(handle, B_SINFO_BUSWIDTH, &bus_width));
  }
  else				/* default for '25 */
  {
    bus_width = 32;
  }

  /* [HL] important: save files only in binary format! */
  if ((fp = fopen(filename, "wb")) == (FILE *) 0)
  {
    B_ERRETURN(B_E_FILE_OPEN);
  }

  rewind(fp);

  data = (b_int32 *) malloc((size_t) (1024 * longs_per_line * 4));

  if (NULL == data)
  {
    B_ERRETURN(B_E_HOST_MEM_FULL);
  }

  lcnt = pretrig > triggerpoint ? 0L : triggerpoint - pretrig;

  if (posttrig < ~0UL)          /* this is the case if we want to limit
                                 * posttrig */
  {
    if (pretrig < ~0UL)
      length = min(lcnt + posttrig + pretrig, length);
    else
      length = min(posttrig + triggerpoint, length);
  }

  /* the if-clause is not trivial!! CZ */
  /* whatever you did here Christian, I am sure it was wrong */
  /*  if (posttrig + triggerpoint > posttrig)
    length = posttrig + triggerpoint > length ? length : posttrig + triggerpoint;   this will not work
    */

  dstart = lcnt;

  fprintf(fp, "Trigger: 0x%lx Lines: 0x%lx Status: %ld Hardware: %s Mode: %d SW: %s %s\n",
	  triggerpoint - dstart, length < dstart ? 0 : length - dstart,
	  status32, handle_array[handle].hwinfo.product_string,
	  modebit | (bus_width == 64UL ? 0x10UL : 0),
	  capi_version_string,
	  storemode == B_WFM_COMPRESSED_1 ? B_WFM_COMP_1_STRING : "");

  while (lcnt < length)
  {
    bound = (int) (((length - lcnt) < 1024) ? length - lcnt : 1024);
    err = BestTraceDataGet(handle, lcnt, (b_int32) bound, data);
    if (err != B_E_OK)
    {
      free(data);
      fclose(fp);
      B_ERRETURN(err);
    }
    if(storemode == B_WFM_COMPRESSED_1)
    {
      fwrite((void *)data,1,bound * longs_per_line * 4,fp);
    }
    else /* standard is still B_WFM_STANDARD which means no compression */
    {
      for (i = 0; i < (b_int32) bound; i++)
      {
	for (j = 0; j < longs_per_line; j++)
	  fprintf(fp, "%08lx ", data[(size_t) (i * longs_per_line + j)]);
	fprintf(fp, "\n");
      }
    }

    lcnt += bound;

    if (cb)
    {
      if ((*cb) ((int) (100 * (lcnt - dstart) / (length - dstart))))
        lcnt = ~0UL;
    }
  }

  free(data);
  fclose(fp);
  B_ERRETURN(B_E_OK);
}

#endif  /* BEST_FIRMWARE */

#endif  /* E2921A_MIN */
